// $Id: CThread.cpp,v 1.8 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "../Testing/CTrace.hpp"
#include "CThread.hpp"
#include "../Exceptions/CException.hpp"
using Exponent::Testing::CTrace;
using Exponent::Exceptions::CException;
using Exponent::Threading::CThread;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CThread, CCountedObject);

//	===========================================================================
CThread::CThread()
       : m_priority(e_highPriority)
	   , m_threadListener(NULL)
	   , m_threadIsActive(false)
	   , m_threadHandle(NULL)
	   , m_threadCriticalSection(NULL)
{
	EXPONENT_CLASS_CONSTRUCTION(CThread);

	// Default priority
	m_priority = e_normalPriority;

	// Default the listener
	NULL_POINTER(m_threadListener);

	// We have no thread handle
	NULL_POINTER(m_threadHandle);

	// No critical sctions
	NULL_POINTER(m_threadCriticalSection);

	// We are not active
	m_threadIsActive = false;
}

//	===========================================================================
CThread::~CThread()
{
	EXPONENT_CLASS_DESTRUCTION(CThread);

	// From a thread itself, we should always be handling the deletion outside..

	// We need to make sure that we have left the thread...
	if (m_threadHandle && m_threadHandle->m_threadHandle)
	{
		throw CException("Thread handle is still valid", "CThread::~CThread()");
	}

	// Delete the thread handle as needed
	FREE_POINTER(m_threadHandle);

	// Forget the critical section
	FORGET_COUNTED_OBJECT(m_threadCriticalSection);
}

//	===========================================================================
bool CThread::runThread()
{
	m_threadIsActive = true;
	return true;
}

//	===========================================================================
void CThread::stopThread()
{
	// Create the scope to kill this thing..
	CCriticalSectionScope scope(new CCriticalSection);

	// ENd the thread
	this->endThread();
}

//	===========================================================================
bool CThread::isThreadActive() const
{
	return m_threadIsActive;
}

//	===========================================================================
void CThread::registerThreadListener(IThreadListener *listener)
{
	m_threadListener = listener;
}

//	===========================================================================
IThreadListener *CThread::getThreadListener()
{
	return m_threadListener;
}

//	===========================================================================
void CThread::setThreadPriority(const EThreadPriority priority)
{
	// Check its not the same
	if (priority == m_priority || m_threadHandle == NULL || !m_threadIsActive)
	{
		return;
	}

	// Store it
	m_priority = priority;

#ifdef WIN32
	switch(m_priority)
	{
		case e_highPriority:		SetThreadPriority(m_threadHandle->m_threadHandle, THREAD_PRIORITY_ABOVE_NORMAL);		break;
		case e_normalPriority:		SetThreadPriority(m_threadHandle->m_threadHandle, THREAD_PRIORITY_NORMAL);				break;
		case e_lowPriority:			SetThreadPriority(m_threadHandle->m_threadHandle, THREAD_PRIORITY_BELOW_NORMAL);		break;
	}
#else
	// No equivalent for POSIX threads
#endif
}

//	===========================================================================
IThread::EThreadPriority CThread::getThreadPriority() const
{
	return m_priority;
}

//	===========================================================================
void CThread::sleepThread(const long timeInMilliseconds)
{
#ifdef WIN32
	Sleep(timeInMilliseconds);
#else
	usleep(1000 * timeInMilliseconds);
#endif
}

//	===========================================================================
void CThread::endThread()
{
	m_threadIsActive = false;
	CTrace::trace("Ending thread %x", &*this);
}

//	===========================================================================
void CThread::setThreadHandle(IThread::SThreadHandle *threadHandle)
{
	m_threadHandle = threadHandle;
}

//	===========================================================================
IThread::SThreadHandle *CThread::getThreadHandle()
{
	return m_threadHandle;
}

//	===========================================================================
void CThread::setCriticalSection(CCriticalSection *criticalSection)
{
	EXCHANGE_COUNTED_OBJECTS(m_threadCriticalSection, criticalSection);
}